#include "util.h"
#include"mc56f8006.h"


/*******************************************************************************
* void init (void)
*
* Perform basic chip initialization.
*
* In:  None
*
* Out: None
*
********************************************************************************
* Note: Ensure COP is disabled. Initialize all CPU registers, including the
*       stack pointer. Then invoke main().
*******************************************************************************/

void entry_point(void)
{
   // Initialize Vector Base Address, Stack Pointer, and OMR
   asm {
	  moveu.w #>sp_start,r0
	  tfra	  r0,sp
	  bfset	  #OMR_SD,OMR
	  move.w  #vba,x:($F041)
   }
   
   // Initialize core registers
   init_cpu();

   // Initialize ROM-mirrored RAM
   init_ram((int*)&pflash_free, (int*)&mirror_start, (int)&mirror_size);
   
   // Initialize BSS
   init_bss((int*)&bss_start, (int)&bss_size); 

   // Configure OCCS
#ifndef NO_LOAD_TRIM_BITS
   PMC_CR2 = FM_OPT0;
#endif
#ifndef ROSC
   set_clk_external();
#endif

   // Execute application code, beginning with "main()"
   main();

   // End simulation with pass/fail signature on GPIO Port
   exit_test();

   // Use an infinite loop to prevent further code execution.
   while(1){};
}


/*******************************************************************************
* void init_cpu (void)
*
* Initialize all CPU registers.
*
* In:  None
*
* Out: None
*
********************************************************************************
* Note: This is desirable in the simulation world for two simple reasons:
*          
*       1.) CPU registers will be set to known values.
*       2.) AGU addressing mode will be initialized to a known state.
*******************************************************************************/
asm void init_cpu(void)
{
        // Initialize ALU registers
        clr     a
        clr     b
        tfr     a,c
        tfr     a,d
        clr.w   x0
        clr.w   y0
        clr.w   y1
        
        // Initialize AGU Registers
        clr.w   r0
        clr.w   r1
        clr.w   r2
        clr.w   r3
        clr.w   r4
        clr.w   r5
        clr.w   n
        clr.w   n3

#if 0
        // The following are no longer needed (SR and M01 are initialized at reset)
        moveu.w #$ffff,m01
        moveu.w #$0300,sr
#endif
        
#if 0
        // Initialize Shadow Registers (needed when using Fast Interrupts)
        swap shadows
        moveu.w #$ffff,m01
        clr.w   r0
        clr.w   r1
        clr.w   n
#endif
        
#if 0
        // Initialize hardware stack
        //
        // Note: Use care to avoid an overflow.  The "do" loop is needed to
        //       initialize the 2nd level in the hardware stack.  At least 3
        //       NOPs are needed between the "move" instruction and "do" loop.
        moveu.w #$9000,la
        moveu.w #$0009,lc
        nop
        nop
        moveu.w #$0,hws
        moveu.w hws,r0
        nop
        nop
        nop
        do      #9,>init_hws
        nop
init_hws:
        nop
#endif

        rts
}


/*******************************************************************************
* void init_bss (register *paddr, register *xaddr, register num)
*
* Initialize BSS data memory space at runtime.
*
* In:  xaddr - (r2) RAM destination address
*      num   - (y0) Number of words to write zero
*
* Out: None
* 
********************************************************************************
* Note: 
*******************************************************************************/
asm void init_bss(register *xaddr, register num)
{
        rep     num
        clr.w   x:(xaddr)+
        rts
}


/*******************************************************************************
* void init_ram (register *paddr, register *xaddr, register num)
*
* Uses PFLASH to initialize constant data memory space at runtime.
*
* In:  paddr - (r2) Flash source address
*      addr  - (r3) RAM destination address
*      num   - (y0) Number of words to copy
*
* Out: None
* 
********************************************************************************
* Note: 
*******************************************************************************/
asm void init_ram(register *paddr, register *xaddr, register num)
{
        do      num,>next_flash_word
        move.w  p:(paddr)+,x0           // Copy PFLASH to X0
        move.w  x0,x:(xaddr)+           // Copy X0 to URAM
        nop                             // Stall (needed for HW loop)
        nop                             // Stall (needed for HW loop)
next_flash_word:
        rts
}


/*******************************************************************************
* void exit_test(void)
*
* Application execution exit point (executed on return from "main()")
* 
* In:  n/a
*
* Out: n/a
*******************************************************************************/
void exit_test(void)
{
#ifndef NO_SIGNATURE
   // Test Application Exit Process
   MB_PUT16(SET_GPIO, PIN_GPIOD0|DRV_OFF);  // Disable PTD3 pin driver
   MB_PUT16(SET_GPIO, PIN_GPIOD1|DRV_OFF);  // Disable PTD2 pin driver
   MB_PUT16(SET_GPIO, PIN_GPIOD2|DRV_OFF);  // Disable PTD1 pin driver
   MB_PUT16(SET_GPIO, PIN_GPIOD3|DRV_OFF);  // Disable PTD0 pin driver
   GPIOD_DDR |= BIT3|BIT2|BIT1|BIT0;        // Use PTD[3:0] GPIO outputs
   GPIOD_DR =  fail_status ? 0xA: 0x5;      // Set PTD[3:0] (0xA=fail, 0x5=pass)
   GPIOD_PER &= ~(BIT3|BIT2|BIT1|BIT0);     // Enable GPIO control on PTD[3:0]
#endif
   EXIT_PORTAL = (uint32_t) error_counter;  // Write EXIT_PORTAL w/ error count
}


/*******************************************************************************
* void init_periphs(void)
*
* Miscellaneous system initialization routines.
* 
* In:  n/a
*
* Out: n/a
*******************************************************************************/
void init_periphs(void)
{
}


/*******************************************************************************
* void set_clk_external()
*
* Switch from ROSC to external clock source (CLKIN or XTAL)
* 
* In:  n/a
*
* Out: n/a
*******************************************************************************/
void set_clk_external(void)
{
   // Call "set_clk_external()" to bypass the internal ROSC 8MHz clock and
   // switch to CLKIN clk 64MHz to get 32MHz ipg_clk
#ifdef EXT_CLK_CLKIN    
   MB_PUT8(MB_SW_EXT_CLK, 0x1);
   // Changing IFE to '0' to reduce PAD delays for supporting 64MHz clock on the CLKIN pin
   GPIOB_IFE &= ~GPIOB_IFE_IFE_6;
   // Change GPIO Peripheral Select to CLKIN
   SIM_GPSB1 |= SIM_GPSB1_GPS_B6;
   // Enabling GPIOB6 (CLKIN) pin by writing into GPIOB_PER register
   GPIOB_PER |= GPIOB_PER_PER_6;
   // Set EXT_SEL to 1 to select CLKIN
   OCCS_OCTRL |= OCCS_OCTRL_EXT_SEL;
   // Changing PRECS to '1' to select the external clk source (direct) bypassing the COSC and ROSC
   OCCS_CTRL |= OCCS_CTRL_PRECS;
#endif
    
   // Call "set_clk_external()" to bypass the internal ROSC 8MHz clock and
   // switch to XTAL clk 64MHz to get 32MHz ipg_clk
#ifdef EXT_CLK_XTAL    
   MB_PUT8(MB_SW_EXT_CLK, 0x2);
   // Changing IFE to '0' to reduce PAD delays for supporting 64MHz clock on the XTAL pin
   GPIOF_IFE &= GPIOF_IFE_IFE_0;
   // Enabling GPIOF0 (XTAL) pin by writing into GPIOF_PER register
   GPIOF_PER = GPIOF_PER_PER_0;
   // Set EXT_SEL to 0 to select XTAL
   OCCS_OCTRL &= ~OCCS_OCTRL_EXT_SEL;
   // Changing PRECS to '1' to select the external clk source (direct) bypassing the COSC and ROSC
   OCCS_CTRL |= OCCS_CTRL_PRECS;
#endif
}

/*******************************************************************************
FUNCTION        : enable_pll
PURPOSE         : enables the pll
INPUTS NOTES    : 
RETURNS NOTES   : none
GENERAL NOTES   : 1) After configuring and enabling the PLL, this routine will loop
*       			 forever waiting for lock 0 detection. Hence, if lock 0 detect never
*       			 occurs, this routine will loop indefinitely.
*    			  2) As suggested by item 2, lock detect interrupts are not used here.
*    			  3) Prescaler, postscaler, and divide by fields are not affected by
*       			 this function. All default values are used. Manipulate these fields
*       			 in order to change the frequency output.
*******************************************************************************/
void enable_pll (void)
{	
	uint16_t temp;
	
	// Select prescaler clock (ZSRC=1), power down PLL, enable lock detector
	temp = OCCS_CTRL;	
	temp &= ~OCCS_CTRL_ZSRC;	
	temp |= (OCCS_CTRL_ZSRC_0 | OCCS_CTRL_PLLPD | OCCS_CTRL_LCKON);	
	OCCS_CTRL = temp;
	
	// Power on PLL, wait for lock (LCK0=1), select postscaler clock (ZSRC=2)
	OCCS_CTRL &= ~OCCS_CTRL_PLLPD;
	
        // Loop until OCCS_STAT[LCK0] = 1
        while (!bittest(OCCS_STAT, OCCS_STAT_LCK0)){}
	
	// Set ZSRC bits
        OCCS_CTRL &= ~OCCS_CTRL_ZSRC;	
	OCCS_CTRL |= OCCS_CTRL_ZSRC_1;
}


/* End of "cpu.c" */
